using System;
using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using SimpleJSON;
using System.Linq;

namespace MVRPlugin 
{
	/// <summary>
	/// Plugin Settings Script for VAMTranslations
	/// </summary>
    public class SettingsVAMTranslations : MVRScript
    {
		
		
		// Variable to permanently store settings 
		private JSONStorableStringChooser JSONAvailableLanguages = null;
		
		// Attribute to be passed on to invoking Plugin
		public                  string    ActiveLanguage         = "English";	
		
		private readonly        string    LanguageEnglish        = "English";
		private readonly        string    LanguageGerman         = "German";
		private readonly        string    LanguageFrench         = "French";
		private readonly        string    LanguageSpanish        = "Spanish";
		private readonly        string    LanguageItalian        = "Italian";
		private readonly        string    LanguagePortugese      = "Portugese";
		private readonly        string    LanguagePolish         = "Polish";
		private readonly        string    LanguageCzech          = "Czech";
		private readonly        string    LanguageRussian        = "Russian";
		private readonly        string    LanguageChinese        = "Chinese";
		private readonly        string    LanguageJapanese       = "Japanese";
		
		private const           string    LabelLanguageEnglish   = "English";
		private const           string    LabelLanguageGerman    = "German - Deutsch";
		private const           string    LabelLanguageFrench    = "French - Français";
		private const           string    LabelLanguageSpanish   = "Spanish - Español";
		private const           string    LabelLanguageItalian   = "Italian - Italiano";
		private const           string    LabelLanguagePortugese = "Portugese - Português";
		private const           string    LabelLanguagePolish    = "Polish - Polski";
		private const           string    LabelLanguageCzech     = "Czech - Čeština";
		private const           string    LabelLanguageRussian   = "Russian - Русский";
		private const           string    LabelLanguageChinese   = "Chinese - 中国人";
		private const           string    LabelLanguageJapanese  = "Japanese - 日本";
		
		/// <summary>
		/// Initialize MVRScript and UI
		/// </summary>
        public override void Init()
        {
            try
            {
				//System.IO.File.WriteAllLines("_SettingsVAMTranslationsPath.txt", new List<string>().ToArray());
				
				// Make sure, that Plugin was called correctly
				if (containingAtom.type != "CoreControl" && containingAtom.type != "SessionPluginManager")
                {
                    SuperController.LogError($"Please load VAMTranslations as a Session or Scene Plugin, not with a '{containingAtom.type}' atom.");
                    DestroyImmediate(this);
                    return;
                }
				
				// Add Caption
				SetupInfoText(this, "<color=#606060><size=40><b>VAM Translations</b></size></color>", 20.0f, false);
				
				// Add informational Notification
				SetupInfoText(this,
                    "<color=#606060><size=40><b>Saving Settings</b></size>\n" +
                    "To permanently save the settings below when restarting Virt-a-Mate, " +
                    "make sure to choose 'Session Plugin Presets -> Change User Defaults -> Set Current As User Defaults'.</color>\n\n",
                    270.0f, false
                );
				
				// Create list of available Languages to choose from
				List<string> availableLanguages = new List<string>()
				{
					LabelLanguageEnglish,  
					LabelLanguageGerman,   
					LabelLanguageFrench,   
					LabelLanguageSpanish,  
					LabelLanguageItalian,  
					LabelLanguagePortugese,
					LabelLanguagePolish,   
					LabelLanguageCzech,    
					LabelLanguageRussian,  
					LabelLanguageChinese,  
					LabelLanguageJapanese 
				};
				
				// Create contents for Drop Down List
				JSONAvailableLanguages = new JSONStorableStringChooser("VAMTranslation_ActiveLanguage", availableLanguages, LabelLanguageEnglish, "Choose Language", OnLanguageChosen);
				
				// Save Active Language permanently
				RegisterStringChooser(JSONAvailableLanguages);
				
				// Create Drop Down UI Element
				CreatePopup(JSONAvailableLanguages).popupPanelHeight = 700f;
            }
            catch (Exception e)
            { SuperController.LogError("Failed to initialize plugin: " + e); }
        }
		
		/// <summary>
		/// Set active Language whenever the Language was changed in the drop down menu
		/// </summary>
		protected void OnLanguageChosen(string ChosenLanguage)
        {
			// Store Language name without native label
			switch (ChosenLanguage)
			{
				case     LabelLanguageEnglish:   ActiveLanguage = LanguageEnglish;   break;
				case     LabelLanguageGerman:    ActiveLanguage = LanguageGerman;    break;
				case     LabelLanguageFrench:    ActiveLanguage = LanguageFrench;    break;
				case     LabelLanguageSpanish:   ActiveLanguage = LanguageSpanish;   break;
				case     LabelLanguageItalian:   ActiveLanguage = LanguageItalian;   break;
				case     LabelLanguagePortugese: ActiveLanguage = LanguagePortugese; break;
				case     LabelLanguagePolish:    ActiveLanguage = LanguagePolish;    break;
				case     LabelLanguageCzech:     ActiveLanguage = LanguageCzech;     break;
				case     LabelLanguageRussian:   ActiveLanguage = LanguageRussian;   break;
				case     LabelLanguageChinese:   ActiveLanguage = LanguageChinese;   break;
				case     LabelLanguageJapanese:  ActiveLanguage = LanguageJapanese;  break;
				default:                         ActiveLanguage = LanguageEnglish;   break;
			}
		}
		
		/// <summary>
		/// Show informational text
		/// </summary>
		private static JSONStorableString SetupInfoText(MVRScript script, string text, float height, bool rightSide)
        {
            JSONStorableString storable = new JSONStorableString("Info", text);
            UIDynamic textfield = script.CreateTextField(storable, rightSide);
            textfield.height = height;
            return storable;
        }

    }
}
